/*
 Note: Currently the abstract Model and Collection are nearly identical and could have easily used
 a mixin to build both in a more DRY way.  I chose not to do this, so that it would be easier for the two
 to diverge encase any future changes dictated a need for that.
 */
define([
    'underscore',
    'backbone',
    'app-resources',
], function(_, Backbone, resources) {
    'use strict';


    /**
     * Creating a model to hold some of the core fetching functions so in order to write more DRY code.
     */
    return Backbone.Model.extend({

        /**
         * Default values for all requests
         */
        _requestDefaults: {
            contentType: 'application/json',
            dataType: 'json',
            traditional: true,
        },

        /**
         * Modifying the Model constructor to ensure `this` is always bound into the url when it is a function
         * and creating custom events for to identify the type of communication with the server
         * @Override
         */
        constructor: function() {
            Backbone.Model.prototype.constructor.apply(this, arguments);
            if (_.isFunction(this.url)) {
                this.url = _.bind(this.url, this);
            }
            this.listenTo(this, 'sync', this._success);
            this.listenTo(this, 'error', this._error);
        },

        /**
         * Overriding the Sync function to include the default options
         *
         * @param {String} method
         * @param {Backbone.Model} model
         * @param {*} options
         * @return {*}
         * @Override
         */
        sync: function(method, model, options) {
            var _options;

            if (!options.method && method === 'create') {
                options.method = 'POST';
            }
            _options = _.extend({}, this._requestDefaults, options);
            return Backbone.Model.prototype.sync.call(this, method, model, _options);
        },

        /**
         * A shortcut to look up resources without have to always include app-resources in the model
         * @param {string} name
         * @return {*}
         */
        getResource: function(name) {
            var _name = name || this.resourceName;
            return resources.get(_name);
        },

        /**
         * Helper function for triggering different responses to the completing a network request
         * @param {string} eventStatus
         * @param {Backbone.Model} model
         * @param {*} response
         * @param {object} options
         * @return {void}
         * @private
         */
        _triggerNetworkResponseEvent: function(eventStatus, model, response, options) {
            var _options = options || {};

            var requestType = _options.method || 'GET';
            var type = requestType.toUpperCase() === 'GET' ? 'fetch' : 'save';
            var message = type + ':' + eventStatus;

            this.trigger(message, model, response, _options);
        },

        /**
         * Creating a unique success event for after communicating with the server.
         *      this.fetch() fires a fetch:success event
         *      this.save() fires a save:success event
         *
         * @param {Backbone.Model} model
         * @param {*} resp
         * @param {*} options
         * @return {void}
         * @private
         */
        _success: function(model, resp, options) {
            this._triggerNetworkResponseEvent('success', model, resp, options);
        },

        /**
         * Creating a unique error event for after communicating with the server.
         *      this.fetch() fires a fetch:error event
         *      this.save() fires a save:error event
         *
         * @param {Backbone.Model} model
         * @param {*} resp
         * @param {*} options
         * @return {void}
         * @private
         */
        _error: function(model, resp, options) {
            this._triggerNetworkResponseEvent('error', model, resp, options);
        },
    });
});
